<?php
/*
 * Plugin Name: StarpayBD
 * Plugin URI: https://securecheckoutio.com/
 * Description: Pay with bKash, Nagad, Rocket and all Bangladeshi gateways via StarpayBD.
 * Author: StarpayBD
 * Version: 1.0.0
 * Requires at least: 5.2
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * Text Domain: starpaybd
 */

add_action('plugins_loaded', 'starpaybd_init_gateway_class');

function starpaybd_init_gateway_class()
{
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_StarpayBD_Gateway extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'starpaybd';
            $this->icon = 'https://starpaybd.com/logo.png';
            $this->has_fields = false;
            $this->method_title = __('StarpayBD', 'starpaybd');
            $this->method_description = __('Pay with bKash, Nagad, Rocket via StarpayBD', 'starpaybd');

            $this->supports = array('products');

            $this->init_form_fields();
            $this->init_settings();

            $this->title       = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled     = $this->get_option('enabled');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_wc_starpaybd_gateway', array($this, 'handle_webhook'));
        }

        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title'   => 'Enable/Disable',
                    'type'    => 'checkbox',
                    'label'   => 'Enable StarpayBD',
                    'default' => 'no'
                ),
                'title' => array(
                    'title'   => 'Title',
                    'type'    => 'text',
                    'default' => 'StarpayBD Payment Gateway',
                ),
                'description' => array(
                    'title'   => 'Description',
                    'type'    => 'textarea',
                    'default' => 'Pay securely using bKash, Nagad, Rocket via StarpayBD.',
                ),
                'apikey' => array(
                    'title'   => 'API Key',
                    'type'    => 'text',
                ),
                'currency_rate' => array(
                    'title'   => 'USD to BDT Rate',
                    'type'    => 'number',
                    'default' => '110',
                ),
                'is_digital' => array(
                    'title'   => 'Digital Product',
                    'type'    => 'checkbox',
                    'label'   => 'Auto complete digital products',
                    'default' => 'no'
                ),
                'payment_site' => array(
                    'title'   => 'Payment Site',
                    'type'    => 'text',
                    'default' => 'https://securecheckoutio.com/',
                    'custom_attributes' => array(
                        'readonly' => 'readonly'
                    ),
                ),
            );
        }

        public function process_payment($order_id)
        {
            $order = wc_get_order($order_id);
            $user  = wp_get_current_user();

            $total = $order->get_total();
            if ($order->get_currency() === 'USD') {
                $total *= $this->get_option('currency_rate');
            }

            $order->update_status('pending', __('Redirecting to StarpayBD...', 'starpaybd'));

            $data = array(
                "cus_name"    => $user->display_name,
                "cus_email"   => $user->user_email,
                "amount"      => round($total, 2),
                "webhook_url" => site_url('/?wc-api=wc_starpaybd_gateway&order_id=' . $order_id),
                "success_url" => $this->get_return_url($order),
                "cancel_url"  => wc_get_checkout_url(),
            );

            $response = $this->api_request(
                'api/payment/create',
                $data
            );

            if (!empty($response['payment_url'])) {
                return array(
                    'result'   => 'success',
                    'redirect' => $response['payment_url']
                );
            }

            wc_add_notice(__('Payment initialization failed', 'starpaybd'), 'error');
            return;
        }

        private function api_request($endpoint, $data)
        {
            $args = array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'API-KEY'      => $this->get_option('apikey'),
                ),
                'body'    => json_encode($data),
                'timeout' => 60,
            );

            $response = wp_remote_post(
                $this->get_option('payment_site') . $endpoint,
                $args
            );

            return json_decode(wp_remote_retrieve_body($response), true);
        }

        public function handle_webhook()
        {
            $order_id = intval($_GET['order_id'] ?? 0);
            $transactionId = $_REQUEST['transactionId'] ?? '';

            $order = wc_get_order($order_id);
            if (!$order) exit;

            $verify = $this->api_request(
                'api/payment/verify',
                array("transaction_id" => $transactionId)
            );

            if (!empty($verify['status']) && $verify['status'] === 'COMPLETED') {

                if ($this->get_option('is_digital') === 'yes') {
                    $order->update_status('completed');
                } else {
                    $order->update_status('processing');
                }

                $order->payment_complete($transactionId);
                $order->add_order_note('StarpayBD payment completed. TXN: ' . $transactionId);
            }

            status_header(200);
            echo 'OK';
            exit;
        }
    }

    add_filter('woocommerce_payment_gateways', function ($gateways) {
        $gateways[] = 'WC_StarpayBD_Gateway';
        return $gateways;
    });
}
